"use client";
import React from "react";
import Image from "next/image";
import { Container } from "../ui/container";
import { TypographyH4, TypographySmall } from "../ui/typography";
import { Button } from "../ui/button";
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuTrigger,
} from "../ui/dropdown-menu";
import { reviews } from "@/data/reviews";

const Stars = ({ rating }: { rating: number }) => {
    const full = Math.floor(rating);
    const half = rating - full >= 0.5;
    return (
        <div className="flex items-center gap-[6.50px]">
            {Array.from({ length: 5 }).map((_, i) => {
                const isFull = i < full;
                const isHalf = i === full && half;
                const src = isFull
                    ? "/assets/icons/full-star.svg"
                    : isHalf
                        ? "/assets/icons/half-star.svg"
                        : "/assets/icons/full-star.svg";
                const cls =
                    isFull || isHalf
                        ? "w-[22.58px] h-[22.58px]"
                        : "w-[22.58px] h-[22.58px] opacity-30";
                return (
                    <Image
                        key={i}
                        src={src}
                        alt="star"
                        width={22.58}
                        height={22.58}
                        className={cls}
                    />
                );
            })}
        </div>
    );
};

const RatingAndReviews: React.FC = () => {
    type SortMode = "Latest" | "Top Rated" | "Lowest Rated";
    const [sortMode, setSortMode] = React.useState<SortMode>("Latest");
    const [visible, setVisible] = React.useState<number>(6);

    const filtered = React.useMemo(() => {
        const list = reviews.slice();
        switch (sortMode) {
            case "Top Rated":
                list.sort((a, b) => b.rating - a.rating);
                break;
            case "Lowest Rated":
                list.sort((a, b) => a.rating - b.rating);
                break;
            default:
                break;
        }
        return list;
    }, [sortMode]);

    const toShow = filtered.slice(0, visible);
    const canLoadMore = visible < filtered.length;

    return (
        <div className="w-full">
            <Container>
                {/* Toolbar */}
                <div className="flex items-start sm:items-center justify-between sm:flex-row flex-col gap-5 mb-6">
                    <div className="text-[22px] sm:text-[24px] leading-[29px] font-bold">
                        All Reviews{" "}
                        <span className="text-[#00000099] text-[16px] font-normal">
                            {" "}
                            ({filtered.length})
                        </span>
                    </div>

                    <div className="flex items-center flex-wrap gap-2.5">
                        <Button
                            variant="outline"
                            className="rounded-full bg-[#F0F0F0] p-3 border-none sm:w-12 w-11 h-11 sm:h-12 hover:bg-[#E8E8E8]"
                        >
                            <Image
                                src="/assets/icons/filter.svg"
                                alt="filter"
                                width={24}
                                height={24}
                                className="w-full h-full"
                            />
                        </Button>

                        {/* Sort */}
                        <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                                <Button
                                    variant="outline"
                                    className="rounded-full h-11 sm:h-12 px-5! gap-0 bg-[#F0F0F0] border-none text-[16px] leading-[22px] font-medium hover:bg-[#E8E8E8] text-black"
                                    aria-label="Change sort"
                                >
                                    {sortMode}
                                    <svg
                                        width="16"
                                        height="16"
                                        className="ml-5"
                                        viewBox="0 0 16 16"
                                        fill="none"
                                        xmlns="http://www.w3.org/2000/svg"
                                    >
                                        <path
                                            d="M13.5306 6.53061L8.5306 11.5306C8.46092 11.6005 8.37813 11.656 8.28696 11.6939C8.1958 11.7317 8.09806 11.7512 7.99935 11.7512C7.90064 11.7512 7.8029 11.7317 7.71173 11.6939C7.62057 11.656 7.53778 11.6005 7.4681 11.5306L2.4681 6.53061C2.3272 6.38972 2.24805 6.19862 2.24805 5.99936C2.24805 5.80011 2.3272 5.60901 2.4681 5.46811C2.60899 5.32722 2.80009 5.24806 2.99935 5.24806C3.19861 5.24806 3.3897 5.32722 3.5306 5.46811L7.99997 9.93749L12.4693 5.46749C12.6102 5.32659 12.8013 5.24744 13.0006 5.24744C13.1999 5.24744 13.391 5.32659 13.5318 5.46749C13.6727 5.60838 13.7519 5.79948 13.7519 5.99874C13.7519 6.198 13.6727 6.38909 13.5318 6.52999L13.5306 6.53061Z"
                                            fill="black"
                                        />
                                    </svg>
                                </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                                {(["Latest", "Top Rated", "Lowest Rated"] as const).map(
                                    (opt) => (
                                        <DropdownMenuItem
                                            key={opt}
                                            onClick={() => setSortMode(opt)}
                                            className={opt === sortMode ? "bg-[#F0F0F0]" : ""}
                                        >
                                            {opt}
                                        </DropdownMenuItem>
                                    )
                                )}
                            </DropdownMenuContent>
                        </DropdownMenu>

                        {/* Write a Review */}
                        <Button className="rounded-full sm:w-fit w-full h-11 sm:h-12 px-[29.5px] text-[16px] leading-[22px] font-medium">
                            Write a Review
                        </Button>
                    </div>
                </div>

                {/* Reviews grid */}
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-4 lg:gap-x-5 lg:gap-y-[64.42px]">
                    {toShow.map((r) => (
                        <div
                            key={r.id}
                            className="sm:py-7 p-6 sm:px-8 border border-[#0000001A] rounded-[20px]"
                        >
                            <div className="flex items-start justify-between">
                                <Stars rating={r.rating} />
                                <Button variant="ghost" className="p-0 hover:bg-transparent">
                                    <Image
                                        src="/assets/icons/dots.svg"
                                        alt="dots"
                                        height={24}
                                        width={24}
                                    />
                                </Button>
                            </div>

                            <div className="mt-[15px] mb-3 flex items-center gap-2">
                                <TypographyH4 className="font-bold text-[20px] leading-3.5">
                                    {r.name}
                                </TypographyH4>
                                {r.verified ? (
                                    <Image
                                        src="/assets/icons/verified.svg"
                                        alt="verified"
                                        width={24}
                                        height={24}
                                    />
                                ) : null}
                            </div>

                            <TypographySmall className="text-[#00000099] text-[16px] leading-[22px] font-normal">
                                “{r.text}”
                            </TypographySmall>

                            <TypographyH4 className="mt-6 text-[#00000099] text-[16px] leading-[22px] font-medium">
                                Posted on {r.date}
                            </TypographyH4>
                        </div>
                    ))}
                </div>

                <div className="mt-20 flex justify-center">
                    <Button
                        variant="outline"
                        className="rounded-full h-[52px] px-[36.5px] text-[16px] leading-5 border border-[#000000] disabled:cursor-not-allowed"
                        disabled={!canLoadMore}
                        onClick={() => setVisible((v) => v + 6)}
                    >
                        {canLoadMore ? "Load More Reviews" : "No more reviews"}
                    </Button>
                </div>
            </Container>
        </div>
    );
};

export default RatingAndReviews;
